<?php

/**
 * Created by Reliese Model.
 */

namespace App\Models;

use Carbon\Carbon;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\DB;

/**
 * Class CustomerDepositsHistory
 *
 * @property int $id
 * @property string $deposit_number
 * @property int|null $customer_id
 * @property int|null $warehousestore_id
 * @property string $department
 * @property float $amount
 * @property int|null $created_by
 * @property int|null $last_updated_by
 * @property int|null $payment_method_id
 * @property Carbon $deposit_date
 * @property Carbon $deposit_time
 * @property string|null $description
 * @property Carbon|null $created_at
 * @property Carbon|null $updated_at
 *
 * @property User|null $user
 * @property Customer|null $customer
 * @property PaymentMethod|null $payment_method
 * @property Warehousestore|null $warehousestore
 *
 * @package App\Models
 */
class CustomerDepositsHistory extends Model
{
	protected $table = 'customer_deposits_history';

	protected $casts = [
		'customer_id' => 'int',
		'warehousestore_id' => 'int',
		'amount' => 'float',
		'created_by' => 'int',
		'last_updated_by' => 'int',
		'payment_method_id' => 'int'
	];

	protected $dates = [
		'deposit_date',
		'deposit_time'
	];

	protected $fillable = [
		'deposit_number',
		'customer_id',
		'warehousestore_id',
		'department',
		'amount',
		'created_by',
		'last_updated_by',
		'payment_method_id',
		'deposit_date',
		'deposit_time',
		'description'
	];


    public static $fields = [
        'deposit_number',
        'customer_id',
        'warehousestore_id',
        'department',
        'amount',
        'created_by',
        'last_updated_by',
        'payment_method_id',
        'deposit_date',
        'deposit_time',
        'description'
    ];


    public function getInvoicePaperNumberAttribute()
    {
        return "DEPOSIT-".$this->deposit_number;
    }

    protected static function boot()
    {
        parent::boot(); // TODO: Change the autogenerated stub

        self::creating(function($model){
            $model->warehousestore_id = getActiveStore()->id;
            $model->department = auth()->user()->department;
            $model->created_by = auth()->id();
            $model->last_updated_by = auth()->id();
            $model->deposit_time = Carbon::now()->toTimeString();
            $model->deposit_number = time();
        });

        self::created(function ($model){

            $model->invoice_number = $model->deposit_number;
            $model->sub_total = $model->amount;

            Payment::createPayment(
                [
                    'invoice' => $model,
                    'type' => "Deposit",
                    'payment_info' => request()->get("payment_info")
                ]
            );

        });

        self::updating(function($model){
            $model->last_updated_by = auth()->id();

        });

        self::updated(function($model){
            DB::statement('SET FOREIGN_KEY_CHECKS=0;');
            $model->paymentMethodTable()->delete();
            $model->payment()->delete();
            DB::statement('SET FOREIGN_KEY_CHECKS=1;');

            $model->invoice_number = $model->deposit_number;
            $model->sub_total = $model->amount;

            Payment::createPayment(
                [
                    'invoice' => $model,
                    'type' => "Deposit",
                    'payment_info' => request()->payment_method_id
                ]
            );
        });

        self::deleting(function($model){
            DB::statement('SET FOREIGN_KEY_CHECKS=0;');
            $model->paymentMethodTable()->delete();
            $model->payment()->delete();
            DB::statement('SET FOREIGN_KEY_CHECKS=1;');
        });

    }


    public static $validate = [
        'amount' => 'required',
        'customer_id' => 'required',
        'deposit_date' => 'required',
        //'payment_info[payment_method_id]' => 'required',
    ];

	public function user()
	{
		return $this->belongsTo(User::class, 'last_updated_by');
	}

	public function customer()
	{
		return $this->belongsTo(Customer::class);
	}

    public function payment()
    {
        return $this->morphOne(Payment::class,'invoice');
    }

    public function paymentMethodTable()
    {

        return $this->morphMany(PaymentMethodTable::class,'invoice');
    }

	public function payment_method()
	{
		return $this->belongsTo(PaymentMethod::class);
	}

	public function warehousestore()
	{
		return $this->belongsTo(Warehousestore::class);
	}
}
